-- start an Oracle Data Pump export job
DECLARE
  schemaList    VARCHAR2(4000) := ?;           -- single quoted, separate by commas
  dumpfileName  VARCHAR2(200) := ?;            -- dump file name, supporting %U formatting
  fileSize  VARCHAR2(200) := ?;                -- maximum file size if not NULL and not empty
  logfileName  VARCHAR2(200) := ?;             -- log file name
  directoryVariable VARCHAR(200) := ?;         -- Oracle directory object, assumed to exist
  jobName VARCHAR(200) := ?;                   -- name of Oracle job
  parallelism NUMBER := ?;                     -- number of parallel streams, 1 means no parallelism
  waitForJob VARCHAR2(20) := ?;                -- wait for job to finish if not NULL and not empty
  remoteLink VARCHAR2(200) := ?;               -- wait for job to finish if not NULL and not empty
 
  handle            NUMBER;
  status            VARCHAR2(20);
  jobMode           VARCHAR2(20) := 'FULL';
BEGIN
 
  IF schemaList IS NOT NULL THEN
    jobMode := 'SCHEMA';
  END IF;
  
  IF remoteLink IS NOT NULL THEN
    handle := DBMS_DATAPUMP.OPEN(
      operation => 'EXPORT',
      job_mode  => jobMode,
      remote_link  => remoteLink,
      job_name  => jobName
      );
  ELSE
    handle := DBMS_DATAPUMP.OPEN(
      operation => 'EXPORT',
      job_mode  => jobMode,
      job_name  => jobName
      );
  END IF;
 
  IF fileSize IS NOT NULL THEN
    DBMS_DATAPUMP.ADD_FILE(
      handle    => handle,
      filename  => dumpfileName,
      directory => directoryVariable,
      filesize => fileSize,
      reusefile => 1);
  ELSE
    DBMS_DATAPUMP.ADD_FILE(
      handle    => handle,
      filename  => dumpfileName,
      directory => directoryVariable,
      reusefile => 1);
  END IF;
 
  DBMS_DATAPUMP.ADD_FILE(
    handle    => handle,
    filename  => logfileName,
    directory => directoryVariable,
    filetype  => DBMS_DATAPUMP.KU$_FILE_TYPE_LOG_FILE,
    reusefile => 1);
 
  IF schemaList IS NOT NULL THEN
    DBMS_DATAPUMP.METADATA_FILTER(
      handle => handle,
      name   => 'SCHEMA_EXPR',
      value  => 'IN (' || schemaList || ')');
  END IF;
 
  -- when using remote link, filter out STATISTICS to avoid this potential error:
  -- Processing object type SCHEMA_EXPORT/TABLE/TRIGGER
  -- Processing object type SCHEMA_EXPORT/TABLE/STATISTICS/TABLE_STATISTICS
  -- Processing object type SCHEMA_EXPORT/STATISTICS/MARKER
  -- ORA-39126: Worker unexpected fatal error in KUPW$WORKER.STATS_UNLOAD [TABLE_STATISTICS] MARKER
  -- ORA-20000: dblink DAOPDBLINK is not compatible for the statistics operation
  IF remoteLink IS NOT NULL THEN
    DBMS_DATAPUMP.METADATA_FILTER(handle, 'EXCLUDE_PATH_EXPR', 'IN (''STATISTICS'')');
  END IF;

  if parallelism <> 1 THEN
    DBMS_DATAPUMP.SET_PARALLEL(handle => handle, degree => parallelism);
  END IF;

  DBMS_DATAPUMP.START_JOB(handle);

  if waitForJob IS NOT NULL THEN
    DBMS_DATAPUMP.WAIT_FOR_JOB(handle, status);
  END IF;
 
END;
