-- start an Oracle Data Pump import job
DECLARE
  schemaList    VARCHAR2(4000) := ?;           -- single quoted schema name(s), separated by commas
  nameList      VARCHAR2(4000) := ?;           -- single quoted table name(s), separated by commas
  dumpfileName  VARCHAR2(200) := ?;            -- dump file name, supporting %U formatting
  logfileName  VARCHAR2(200) := ?;             -- log file name
  directoryVariable VARCHAR(200) := ?;         -- Oracle directory object, assumed to exist
  jobName VARCHAR(200) := ?;                   -- name of Oracle job
  parallelism NUMBER := ?;                     -- number of parallel streams, 1 means no parallelism
  waitForJob VARCHAR2(20) := ?;                -- wait for job to finish if not NULL and not empty
  overwriteTables VARCHAR2(20) := ?;           -- overwrite existing tables if not NULL and not empty
  remoteLink VARCHAR2(200) := ?;               -- wait for job to finish if not NULL and not empty
 
  handle            NUMBER;
  status            VARCHAR2(20);
  jobMode           VARCHAR2(20) := 'FULL';
  replaceAction     VARCHAR2(20) := 'SKIP';
BEGIN
 
  IF schemaList IS NOT NULL THEN
    jobMode := 'SCHEMA';
  END IF;

  -- in table mode, nameList must be the table simple names (without the schema info),
  -- single quoted and separated by commas.
  -- in this mode, schemaList lists the single schema in which all of the tables are
  -- located.  as with schema mode, the schema list (in this case just a single schema) must be single quoted.
  -- example schemaList to restore a list of tables from HR schema:  'HR'
  -- example nameList (table names):  'EMPLOYEES','DEPARTMENTS
  IF nameList IS NOT NULL THEN
    jobMode := 'TABLE';
  END IF;

  IF remoteLink IS NOT NULL THEN
    handle := DBMS_DATAPUMP.OPEN(
      operation => 'IMPORT',
      job_mode  => jobMode,
      remote_link  => remoteLink,
      job_name  => jobName
      );
  ELSE
    handle := DBMS_DATAPUMP.OPEN(
      operation => 'IMPORT',
      job_mode  => jobMode,
      job_name  => jobName
      );
  END IF;
 
  DBMS_DATAPUMP.ADD_FILE(
    handle    => handle,
    filename  => dumpfilename,
    directory => directoryvariable);
 
  DBMS_DATAPUMP.ADD_FILE(
    handle    => handle,
    filename  => logfileName,
    directory => directoryvariable,
    filetype  => DBMS_DATAPUMP.KU$_FILE_TYPE_LOG_FILE,
    reusefile => 1);
 
  IF overwriteTables IS NOT NULL THEN
    replaceAction := 'REPLACE';
  END IF;

  DBMS_DATAPUMP.SET_PARAMETER(
    handle => handle,
    name   => 'table_exists_action',
    value  => replaceAction);
 
  IF schemaList IS NOT NULL THEN
    DBMS_DATAPUMP.METADATA_FILTER(
      handle => handle,
      name   => 'SCHEMA_EXPR',
      value  => 'IN (' || schemaList || ')');
  END IF;
 
  IF nameList IS NOT NULL THEN
    DBMS_DATAPUMP.METADATA_FILTER(
      handle => handle,
      name   => 'NAME_EXPR',
      value  => 'IN (' || nameList || ')');
  END IF;

  -- java_add_code_here

  if parallelism <> 1 THEN
    DBMS_DATAPUMP.SET_PARALLEL(handle => handle, degree => parallelism);
  END IF;

  DBMS_DATAPUMP.START_JOB(handle);

  if waitForJob IS NOT NULL THEN
    DBMS_DATAPUMP.WAIT_FOR_JOB(handle, status);
  END IF;
 
END;
