# -*- coding: utf-8 -*-

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""Helper file that executes the  Intellisnap basic acceptance test cases for nas client

Intellisnap BasicAcceptance is the only class defined in this file

This class include below cases:
    1.  FULL backup job

    2.  INCREMENTAL backup job after adding test data

    3.  DIFFERENTIAL backup job after adding test data

    4.  Restore out of place to Windows client

    5.  Restore out of place to Unix client

    6.  Restore in place job

    7.  Restore out of place to filer job

    8.  Running backup copy job

    9. Restore in place from backup copy job

    10. Restore in place in incremental job time frame

BasicAcceptance:
    __init__()              --  initializes basicacceptance object

    _get_copy_precedence()  --  returns the copy precedence value

    _run_backup()           --  starts the backup job

    run()                   --  runs the basic acceptance test case
"""


import time

from NAS.NASUtils.nashelper import NASHelper
from NAS.NASUtils.nasclient import NetAPPClient
from AutomationUtils.machine import Machine
from AutomationUtils import logger
from AutomationUtils.options_selector import OptionsSelector


class SnapBasicAcceptance(object):
    """Helper class to run Intellisnap basic acceptance test case for nas client"""

    def __init__(self, test_case_obj, is_cluster=False):
        """Initializes snapbasicacceptance object

            Args:
                test_case_obj   (object)    --  test case class object

                is_cluster      (bool)      --  flag to determine if the specified client
                                                    is cluster / vserver / filer
        """
        self._inputs = test_case_obj.tcinputs
        self._commcell = test_case_obj.commcell
        self._log = logger.get_log()
        self._subclient = test_case_obj.subclient
        self._commserver_name = self._commcell.commserv_name
        self._csdb = test_case_obj.csdb
        self._nas_helper = NASHelper()
        self._client = test_case_obj.client
        self._is_cluster = is_cluster
        self._storage_policy = self._commcell.storage_policies.get(self._subclient.storage_policy)
        self.client_machine = Machine(self._commserver_name, self._commcell)
        self.mount_path = 'tstmntpath'
        self.mountpath_val = None
        self.mount_status = None
        self.test_data_path = None
        self.get_mount_path = "SELECT MountPath FROM SMVolume WHERE JobId = {0}"
        self.get_volume_id = "SELECT SMVolumeId FROM SMVolume WHERE jobID = {0}"
        self.get_mount_status = "SELECT MountStatus FROM SMVolume WHERE JobId = {0}"
        self.client_name = self._client.client_name
        self._agent = test_case_obj.agent
        self.impersonate_user = None
        self.impersonate_password = None
        self.proxy = None

    def _run_backup(self, backup_type):
        """Starts backup job"""
        self._log.info("*" * 10 + " Starting Subclient {0} Backup ".format(backup_type) + "*" * 10)
        job = self._subclient.backup(backup_type)
        self._log.info("Started %s backup with Job ID: %s", backup_type, str(job.job_id))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run {0} backup job with error: {1}".format(
                    backup_type, job.delay_reason
                )
            )
        return job

    def execute_query(self, query, option1):
        """ Executes SQL Queries
            Return:
                    str : first column of the sql output

        """
        self._csdb.execute(query.format(option1))
        return self._csdb.fetch_one_row()[0]

    def execute_query_all_rows(self, query, option1):
        """ Executes SQL Queries
            Args:
                query           (str)   -- sql query to execute

                my_options       (dict)  -- options in the query

            Return:
                    list : all column of the sql output


        count = len(my_options)
        if count == 1:
            self._csdb.execute(query.format(my_options['option1']))
        if count == 2:
            self._csdb.execute(query.format(my_options['option1'], my_options['option2']))
        if count == 3:
            self._csdb.execute(query.format(my_options['option1'], my_options['option2'],
                                            my_options['option3']))"""
        self._csdb.execute(query.format(option1))
        return self._csdb.fetch_all_rows()

    def _get_copy_precedence(self, storage_policy, storage_policy_copy):
        """Returns the copy precedence value"""
        self._csdb.execute(
            "select copy from archGroupCopy where archGroupId in (select id from archGroup where \
            name = '{0}') and name = '{1}'".format(storage_policy, storage_policy_copy))
        cur = self._csdb.fetch_one_row()
        return cur[0]

    def snap_operations(self, jobid, _commserver_name=None, mountpath=None,
                        mount=False,
                        unmount=False,
                        revert=False,
                        delete=False):
        """ Common Method for Snap Operations
            Args :
                jobid : jobid for the Snap operation
                _commserver_name : name of the destination client, default: None
                MountPath : MountPath for Snap operation, default: None
            Return :
                object : Job object of Snap Operation job
        """

        self._log.info("Getting SMVolumeId using JobId:  %s}", format(jobid))
        volumeid = self.execute_query_all_rows(self.get_volume_id, jobid)
        self._log.info("SMvolumeId is : %s", format(volumeid))
        self._log.info("destination client id is :%s", format(_commserver_name))
        self._log.info("mountpath is %s", format(mountpath))
        mountid = []
        mountid.append(volumeid[0])
        if mount:
            job = self._commcell.array_management.mount(mountid,
                                                        _commserver_name,
                                                        mountpath,
                                                        False)
        elif unmount:
            job = self._commcell.array_management.unmount(mountid)
        elif revert:
            job = self._commcell.array_management.revert(mountid)
        else:
            job = self._commcell.array_management.delete(volumeid)

        self._log.info("Started  job : %s for Snap Operation", format(job.job_id))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run   job {0} for Snap operation with error: {1}".format(
                    job.job_id, job.delay_reason)
                )
        if job.status != 'Completed':
            raise Exception(
                "job: {0} for snap operation is completed with errors, Reason: {1}".format(
                    job.job_id, job.delay_reason)
                )

        self._log.info("successfully completed Snap operation of jobid :%s", format(job.job_id))
        time.sleep(30)
        return job

    def mount_snap(self, jobid, destclient=None):
        """ Mounts Snap of the given jobid
            Args:
                jobid      : jobid for mount operation
                destclient : Client name on which snap is to be mounted
            Return:
                object : job object of Snap operation job
        """
        self._log.info("Mounting snapshot of jobid : %s}", format(jobid))
        mountpath = self.mount_path
        if destclient is None:
            destclient = self._commserver_name
        return self.snap_operations(jobid, destclient, mountpath, mount=True)

    def unmount_snap(self, jobid):
        """ UnMounts Snap of the given jobid
            Args:
                jobid : jobid for unmount operation
            Return:
                object : job object of Snap operation job
        """
        self._log.info("UnMounting snapshot of jobid : %s", format(jobid))
        return self.snap_operations(jobid, unmount=True)

    def revert_snap(self, jobid):
        """ Reverts Snap of the given jobid
            Args:
                jobid : jobid for revert operation
            Return:
                object : job object of Snap operation job
        """
        self._log.info("Reverting snapshot of jobid : %s", format(jobid))
        return self.snap_operations(jobid, revert=True)

    def delete_snap(self, jobid):
        """ Deletes Snap of the given jobid
            Args:
                jobid : jobid for delete operation
            Return:
                object : job object of Snap operation job
        """
        self._log.info("Deleting snapshot of jobid : %s", format(jobid))
        return self.snap_operations(jobid, delete=True)

    def snapshot_cataloging(self):
        """ Runs Offline snapshot cataloging for the given storage policy
        """

        job = self._storage_policy.run_snapshot_cataloging()
        self._log.info("Deferred Catalog workflow job id is : {0}".format(job.job_id))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run  Workflow job {0} for Deferred Catalog with error: {1}".format(
                    job.job_id, job.delay_reason)
                )
        if job.status != 'Completed':
            raise Exception(
                "job: {0} for Snapshot catalog operation is completed with errors".format(
                    job.job_id)
                )
        self._log.info("Successfully completed deferred catalog job: {0}".format(job.job_id))

    def compare(self, client_obj, destclient_obj, source_dir, dest_dir):
        """ Compare two directories
            Args:
                client_obj : client Object of the Source machine
                destclient_obj : client Object of the Destination machine
                source_dir : Source directory Path
                dest_dir : Destination directory Path
        """

        self._log.info("comparing content")
        self._log.info("source dir: %s", source_dir)
        self._log.info("dest dir : %s", dest_dir)

        difference = []
        difference = client_obj.compare_folders(
            destclient_obj,
            source_dir,
            dest_dir,
            ignore_files=self._nas_helper.ignore_files_list,
            ignore_folder=self._nas_helper.ignore_files_list
            )

        if difference != []:
            self._log.error(
                "Validation failed. List of different files \n%s", format(difference)
            )
            raise Exception(
                "validation failed. Please check logs for more details."
            )
        self._log.info("Compare folder validation was successful")

    def get_fs_os_id(self):
        """ Returns the iDA type value of the subclient """
        self.scid = self._subclient._get_subclient_id()
        query = "select appTypeId from app_application where id={0}"
        fsosid = self.execute_query(query, self.scid)
        return fsosid

    def snapop_validation(
            self, jobid, mount=False, revert=False, delete=False, unmount=False, destclient=None):
        """ Common Method for Snap Operation Validations
            Args:
                jobid      : snap backup jobid
                mount      : if given true, will do mount validation
                revert     : if given true, will do revert validation
                delete     : if given true, will do delete validation
                unmount    : if given true, will do unmount validation
                destclient : Client object on which snap operations should be done
        """

        self._log.info("validating snap operation")
        fsosid = self.get_fs_os_id()
        if fsosid != '29':
            volume_path, _ = self.nas_client.get_path_from_content(self._subclient.content[0])
            volume_path = str(volume_path)
        else:
            volume_path = str(self._subclient.content[0])
        if mount or revert:
            self.mountpath_val = str(self.execute_query(self.get_mount_path, jobid))
            if destclient is None:
                self.compare(
                    self.client_machine, self.client_machine, self.mountpath_val, volume_path)
            else:
                self.compare(
                    destclient, destclient, self.mountpath_val, volume_path)
            self._log.info("comparing files/folders was successful")

        elif delete:
            self._log.info("Checking if the snapshot of JobId: %s exists in the DB", format(jobid))
            self._log.info(
                "smvolumeid from DB is: %s", format(self.execute_query_all_rows(self.get_volume_id, jobid)))
            if format(self.execute_query_all_rows(self.get_volume_id, jobid)) == format([['']]):
                self._log.info("Snapshot is successfully deleted")
            else:
                raise Exception(
                    "Snapshot of jobid: {0} is not deleted yet, please check the CVMA logs".format(
                        jobid)
                    )
            self._log.info("Successfully verified Snapshot cleanup")

        else:
            self.mountpath_val = str(self.execute_query(self.get_mount_path, jobid))
            if destclient is None:
                if self.client_machine.check_directory_exists(self.mountpath_val):
                    raise Exception("MountPath folder still exists under {0}".format(
                        self.mountpath_val))
                else:
                    self._log.info("MountPath folder does not exists ")
            else:
                if destclient.check_directory_exists(self.mountpath_val):
                    raise Exception("MountPath folder still exists under {0}".format(
                        self.mountpath_val))
                else:
                    self._log.info(
                        "Path %s don't exist on %s", self.mountpath_val, destclient.machine_name)

    def mount_validation(self, jobid, destclient=None):
        """ Mount Snap validation
            Args :
                jobid      : snap backup jobid
                destclient : destination client on which snap is mounted
        """
        self.snapop_validation(jobid, mount=True, destclient=destclient)
        self._log.info("mount validation was successful")

    def revert_validation(self, jobid, destclient=None):
        """ Revert Snap validation
            Args :
                jobid      : snap backup jobid
                destclient : destination client on which snap is to be mounted
        """
        if destclient is not None:
            self.mount_snap(jobid, destclient.machine_name)
        else:
            self.mount_snap(jobid)
        self.snapop_validation(jobid, revert=True, destclient=destclient)
        self.unmount_snap(jobid)
        self._log.info("revert validation was successful")

    def delete_validation(self, jobid):
        """ Delete Snap validation
            Args :
                jobid : snap backup jobid
        """
        self.snapop_validation(jobid, delete=True)
        self._log.info("delete validation was successful")

    def unmount_validation(self, jobid, destclient=None):
        """ UnMount Snap validation
            Args :
                jobid      : snap backup jobid
                destclient : destination client on which snap is to be unmounted
        """
        self.snapop_validation(jobid, unmount=True, destclient=destclient)
        self._log.info("unmount validation was successful")

    def check_mountstatus(self, jobid, unmount=False):
        """ Common function to check mount status
            Args:
                jobid : snap backup jobid
        """

        if unmount:
            while self.mount_status not in ['79', '']:
                self.mount_status = self.execute_query(self.get_mount_status, jobid)
                self._log.info("mount status: jobid :%s is :%s", format(jobid, self.mount_status))
                self._log.info("snapshot is not unmounted yet, checking after 1min")
                time.sleep(60)
                continue
            self._log.info("snapshot of jobid %s is unmounted successfully", format(jobid))

        return self.mount_status

    def unmount_status(self, jobid):
        """ Check Unmount status
            Args:
                jobid : snap backup jobid
        """
        return self.check_mountstatus(jobid, unmount=True)

    def cleanup(self):
        """
            Remove Subclient content
            Remove Restore location
            Remove Mountpath
        """
        volume_path, _ = self.nas_client.get_path_from_content(self._subclient.content[0])
        volume_path = str(volume_path + '\\TestData')
        self.client_machine.remove_directory(volume_path)
        self._log.info("Successfully removed Subclient Content")
        self.client_machine.remove_directory(self.mount_path)
        self._log.info("Successfully removed MountPath")

    def run(self):
        """Executes Intellisnap basic acceptance test case"""
        self._log.info(
            "Will run below test case on: %s subclient", format(str(self._inputs['SubclientName']))
        )

        self._log.info("Number of data readers: " + str(self._subclient.data_readers))
        if self._subclient.data_readers != 3:
            self._log.info("Setting the data readers count to 3")
            self._subclient.data_readers = 3

        self._log.info("Get NAS Client object")
        self.nas_client = self._nas_helper.get_nas_client(self._client, self._agent,
                                                          is_cluster=self._is_cluster)

        self._log.info("Make a CIFS Share connection")
        self.nas_client.connect_to_cifs_share(
            str(self._inputs['CIFSShareUser']), str(self._inputs['CIFSSharePassword'])
        )
        job = self._run_backup("FULL")
        for content in self._subclient.content:
            volume_path, _ = self.nas_client.get_path_from_content(content)
            self._nas_helper.copy_test_data(self.nas_client, volume_path)

        job = self._run_backup("INCREMENTAL")

        inc_job_start_time = str(job.start_time)
        inc_job_end_time = str(job.end_time)
        if self._inputs.get('mount_path'):
            self.mount_path = str(self._inputs['mount_path'])

        self.mount_snap(job.job_id)
        self.mount_validation(job.job_id)
        self.unmount_snap(job.job_id)
        self.unmount_validation(job.job_id)

        for content in self._subclient.content:
            volume_path, _ = self.nas_client.get_path_from_content(content)
            self._nas_helper.copy_test_data(self.nas_client, volume_path)

        job = self._run_backup("DIFFERENTIAL")

        options_selector = OptionsSelector(self._commcell)

        size = self.nas_client.get_content_size(self._subclient.content)
        if self._inputs.get('liveBrowse'):
            if self._inputs['liveBrowse'].upper() == 'TRUE':
                fs_options = {'live_browse': True}
        windows_restore_client, windows_restore_location = \
            options_selector.get_windows_restore_client(size=size)

        self._log.info("*" * 10 + " Run out of place restore to Windows Client " + "*" * 10)

        job = self._subclient.restore_out_of_place(
            windows_restore_client.machine_name,
            windows_restore_location,
            self._subclient.content,
            fs_options=fs_options
        )
        self._log.info(
            "Started Restore out of place to Windows client job with Job ID: " + str(job.job_id)
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: " + str(job.delay_reason)
            )

        self._log.info("Successfully finished Restore out of place to windows client")

        self._nas_helper.validate_windows_restored_content(
            self.nas_client, windows_restore_client, windows_restore_location, \
                self._subclient.content
        )

        self._log.info("*" * 10 + " Run out of place restore to Linux Client" + "*" * 10)

        linux_restore_client, linux_restore_location = \
            options_selector.get_linux_restore_client(size=size)

        job = self._subclient.restore_out_of_place(
            linux_restore_client.machine_name,
            linux_restore_location,
            self._subclient.content,
            fs_options=fs_options
        )
        self._log.info(
            "Started restore out of place to linux client job with Job ID: " + str(job.job_id)
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: " + str(job.delay_reason)
            )

        self._log.info("Successfully finished Restore out of place to linux client")

        out = []
        out = windows_restore_client.compare_folders(
            linux_restore_client, windows_restore_location,
            linux_restore_location, ignore_files=self._nas_helper.ignore_files_list)
        if out != []:
            self._log.error(
                "Restore validation failed. List of different files \n%s", format(str(out))
            )
            raise Exception(
                "Restore validation failed. Please check logs for more details."
            )

        self._log.info("Successfully validated restored content")

        self._log.info("*" * 10 + " Run in place restore in incremental jobtime frame " + "*" * 10)
        job = self._subclient.restore_in_place(
            self._subclient.content,
            from_time=inc_job_start_time,
            to_time=inc_job_end_time,
            fs_options=fs_options)

        self._log.info(
            "Started restore in place in incremental jobtime frame job with Job ID: %s", format(
                job.job_id
            )
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore in incremental time frame with error: {0}".format(
                    job.delay_reason
                )
            )

        self._log.info("Successfully finished Restore in place in incremental time frame")

        self._nas_helper.validate_filer_restored_content(
            self.nas_client, windows_restore_client, windows_restore_location, \
                self._subclient.content
        )

        self._log.info("*" * 10 + " Run Restore in place " + "*" * 10)
        job = self._subclient.restore_in_place(self._subclient.content,
                                               fs_options=fs_options)
        self._log.info("Started restore in place job with Job ID: %s", format(str(job.job_id)))

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore in place job with error: {0}".format(job.delay_reason)
            )

        self._log.info("Successfully finished restore in place job")

        self._nas_helper.validate_filer_restored_content(
            self.nas_client, windows_restore_client, windows_restore_location, \
                self._subclient.content
        )

        self._log.info("*" * 10 + " Run out of place restore to Filer " + "*" * 10)
        filer_restore_location = str(self._inputs['FilerRestoreLocation'])

        job = self._subclient.restore_out_of_place(
            self._client.client_name,
            filer_restore_location,
            self._subclient.content,
            fs_options=fs_options)

        self._log.info(
            "Started Restore out of place to filer job with Job ID: %s}", format(job.job_id)
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: {0}".format(job.delay_reason)
            )

        self._log.info("Successfully finished Restore out of place to Filer")

        self._nas_helper.validate_filer_restored_content(
            self.nas_client, windows_restore_client, windows_restore_location,
            self._subclient.content, filer_restore_location
        )

        storage_policy = self._commcell.storage_policies.get(self._subclient.storage_policy)
        storage_policy_copy = "Primary"

        self._log.info("*" * 10 + "Running backup copy now" + "*" * 10)

        job = self._storage_policy.run_backup_copy()
        self._log.info("Backup copy workflow job id is : %s", format(job.job_id))
        if not job.wait_for_completion():
            raise Exception(
                "Failed to run restore out of place job with error: " + str(job.delay_reason)
            )
        self._log.info("Successfully finished backup copy workflow Job :%s", format(job.job_id))

        if job.status != 'Completed':
            raise Exception(
                "job: {0} for Backup copy operation is completed with errors, Reason: {1}".format(
                    job.job_id, job.delay_reason)
            )

        self._log.info("*" * 10 + " Run in place restore from backup copy " + "*" * 10)
        copy_precedence = self._get_copy_precedence(
            self._subclient.storage_policy, storage_policy_copy
        )

        job = self._subclient.restore_in_place(
            self._subclient.content, copy_precedence=int(copy_precedence)
        )

        self._log.info(
            "Started restore in place from backup copy job with Job ID: %s", format(job.job_id)
        )

        if not job.wait_for_completion():
            raise Exception(
                "Failed to restore from backup copy with error: {0}".format(str(job.delay_reason))
            )

        self._log.info("Successfully finished Restore in place from backup copy")

        self._nas_helper.validate_filer_restored_content(
            self.nas_client, windows_restore_client, windows_restore_location, \
                self._subclient.content
        )
        if self._nas_helper.nas_vendor(self._client) != 'Isilon':
            self._log.info("*" * 10 + " Run deferred cataloging on the storage policy " + "*" * 10)
            self.snapshot_cataloging()

            self._log.info("*" * 10 + "Run out of place restore to Filer from deferred catalog" +
                           "*" * 10)
            storage_policy_copy = "Primary Snap"
            copy_precedence = self._get_copy_precedence(
                self._subclient.storage_policy, storage_policy_copy
            )
            job = self._subclient.restore_out_of_place(self._client.client_name,
                                                       filer_restore_location,
                                                       self._subclient.content,
                                                       copy_precedence=int(copy_precedence),
                                                       fs_options=fs_options)
            self._log.info(
                "Started Restore out of place to filer job with Job ID: %d", job.job_id
            )
            if not job.wait_for_completion():
                raise Exception(
                    "Failed to run restore out of place job with error:%s", job.delay_reason
                )
            self._log.info("Successfully finished Restore out of place to Filer from catalog")
            self._nas_helper.validate_filer_to_filer_restored_content(
                self.nas_client, self._subclient.content, filer_restore_location
            )
        job = self._run_backup("FULL")

        if isinstance(self.nas_client, NetAPPClient):
            volume_path, _ = self.nas_client.get_path_from_content(self._subclient.content[0])
            self._nas_helper.copy_test_data(self.nas_client, volume_path)
            self.revert_snap(job.job_id)
            self.revert_validation(job.job_id)
        time.sleep(60)
        self.delete_snap(job.job_id)
        self.delete_validation(job.job_id)

        self.cleanup()
