# -*- coding: utf-8 -*-
# pylint: disable=W0223
# pylint: disable=W0221
# pylint: disable=R0913

# --------------------------------------------------------------------------
# Copyright Commvault Systems, Inc.
# See LICENSE.txt in the project root for
# license information.
# --------------------------------------------------------------------------

"""File for performing Hadoop operations on a machine with Hadoop Client.

This file consists of a class named: HadoopMachine, which can connect to the Hadoop Cluster
and perform various Hadoop operations.

The instance of this class can be used to perform various operations on a Hadoop Cluster, like,

    #.  Generating test data on hdfs
    #.  Modifying test data on hdfs
    #.  Get information about the items on the given path on hdfs
    #.  Remove an existing Directory on hdfs

HadoopMachine
=============

    __init__()                           --  initialize object of the class

    generate_test_data()                 --  generates and adds random testdata on the specified path

    modify_test_data()                   --  Modifies the test data at the given path

    get_test_data_info()                 --  Gets information about the items on the given path

    get_items_list()                     --  Gets the list of items at the given path

    remove_directory()                   --  Removes a directory on the hadoop client.

    create_directory()                   --  Creates a directory on the hadoop client.

"""

import os
from .unix_machine import UnixMachine
from .constants import UNIX_TMP_DIR
from .constants import HADOOP_MANAGE_DATA


class HadoopMachine(UnixMachine):
    """Class for performing operations on a UNIX OS remote Hadoop client."""

    def __init__(self,
                 hdfs_user,
                 machine_name,
                 commcell_object=None,
                 username=None,
                 password=None,
                 key_filename=None,
                 run_as_sudo=False):
        """Initializes instance of the HadoopMachine class.

        Args:
            hdfs_user           (str)       --  hadoop username for the client to connect to hadoop

            machine_name        (str)       --  name / ip address of the client to connect to

            commcell_object     (object)    --  instance of the Commcell class from CVPySDK

                default:    None

            username            (str)       --  username for the client to connect to

                    Only Applicable if the client is not a Commvault Client

                default:    None

            password            (str)       --  password for the above specified user

                default:    None

            key_filename        (str/list)  --  string or list containing ppk key location(s) for

                    machines that require a private key for SSH

                default:    None

            run_as_sudo         (bool)      --  variable for running commands as sudo for machines

                    where root login is disabled

                default:    False

        Also, initializes the Client object, if it is Commvault Client.

        Otherwise, it creates a paramiko SSH client object for the client.

        """
        super(HadoopMachine, self).__init__(
            machine_name, commcell_object, username, password, key_filename, run_as_sudo)
        self._hdfs_user = hdfs_user
        self._jar_path = self.join_path(UNIX_TMP_DIR, os.path.basename(HADOOP_MANAGE_DATA))
        super(HadoopMachine, self).remove_directory(self._jar_path)
        self.copy_from_local(HADOOP_MANAGE_DATA, UNIX_TMP_DIR)

    def _execute_jar(self, script_arguments=""):
        """Execute the hadoop jar file with given arguments

        Args:
            script_arguments       (str)             --  Arguments to the hadoop jar file

        """
        java_command = """
        export CLASSPATH=`hadoop classpath --glob`;
        java -cp $CLASSPATH:{0} com.cv_hadoop.cv_hdfs_manage_data.ManageHadoopData {1} -user {2} 2>/dev/null
        """.format(self._jar_path, script_arguments, self._hdfs_user)
        return self.execute(java_command)

    def generate_test_data(self, file_path, **kwargs):
        """Generates and adds random test data at the given path with the specified options

        Args:
            file_path           (str)   --  directory path where
                                            the data will be generated.

            kwargs              (dict)  -- dict of keyword arguments as follows

                dirs                (int)   --  number of directories
                                                in each level

                    default: 1

                files               (int)   --  number of files
                                                in each directory

                    default: 1

                file_size           (int)   --  Size of the files in MB

                    default: 512

                levels              (int)   --  number of levels to be created

                    default: 1

                sparse              (bool)  --  whether to create sparse files

                    default: True

                sparse_hole_size    (int)   --  Size of the holes
                                                in sparse files in MB

                    default: 100

                long_path           (bool)  --  whether to create long files

                    default: False

                long_level          (int)   --  length of the long path

                    default: 1200

                unicode             (bool)  --  whether to create
                                                unicode files

                    default: False

        Returns:
            bool    -   boolean value True is returned
                        if no errors during data generation.

        Raises:
            Exception:
                if any error occurred while generating the test data.

        """
        dirs = kwargs.get('dirs', 1)
        files = kwargs.get('files', 1)
        file_size = kwargs.get('file_size', 512)
        levels = kwargs.get('levels', 1)
        sparse = kwargs.get('sparse', True)
        sparse_hole_size = kwargs.get('sparse_hole_size', 100)
        long_path = kwargs.get('long_path', False)
        long_level = kwargs.get('long_level', 1200)
        unicode = kwargs.get('unicode', False)
        script_arguments = (
            "-optype add -path \"{0}\" -regular -dirs {1} -files {2} -sizeinmb {3} -levels {4}".format(
                file_path,
                str(dirs),
                str(files),
                str(file_size),
                str(levels)))

        if sparse:
            script_arguments = ("{0} -sparse"
                                " -holesizeinmb {1}".format(
                                    script_arguments, str(sparse_hole_size)))
        if long_path:
            script_arguments = ("{0} -longpath"
                                " -longlevel {1}".format(
                                    script_arguments, str(long_level)))
        if unicode:
            script_arguments = "{0} -unicode".format(script_arguments)

        output = self._execute_jar(script_arguments)

        if output.exit_code != 0:
            raise Exception(
                "Error occurred while generating test data {0} {1}".format(output.output, output.exception)
            )
        return True

    def modify_test_data(self, data_path, **kwargs):
        """Modifies the test data at the given path based on the specified options

        Args:
            data_path   (str)   --  directory path where the dataset resides.

            kwargs      (dict)  --  dict of keyword arguments as follows

                rename_dir          (bool)  --  whether to rename the whole directory

                    default: False

                rename              (bool)  --  whether to rename all files

                    default: False

                modify              (bool)  --  whether to modify data of all files

                    default: False

                permissions         (bool)  --  whether to change permission of all files

                    default: False

        Returns:
            bool    -   boolean value True is returned
                        if no errors during data modification.

        Raises:
            Exception:
                if any error occurred while modifying the test data.

        """
        rename_dir = kwargs.get('rename_dir', False)
        rename = kwargs.get('rename', False)
        modify = kwargs.get('modify', False)
        permissions = kwargs.get('permissions', False)
        script_arguments = "-optype change -path \"{0}\"".format(data_path)

        if rename_dir:
            script_arguments = "{0} -renamedir".format(script_arguments)

        if rename:
            script_arguments = "{0} -rename".format(script_arguments)

        if modify:
            script_arguments = "{0} -modify".format(script_arguments)

        if permissions:
            script_arguments = "{0} -permissions".format(script_arguments)

        output = self._execute_jar(script_arguments)

        if output.exit_code != 0:
            raise Exception(
                "Error occurred while modifying test data {0} {1}".format(output.output, output.exception)
            )
        return True

    def get_test_data_info(self, data_path, **kwargs):
        """Gets information about the items on the given path based on the given options

        Args:
            data_path             (str)   --  directory path from where the data should be retrieved.

            kwargs                (dict)  --  dict of keyword arguments as follows

                meta                  (bool)  --  whether to get meta data of all files

                    default: False

                checksum              (bool)  --  whether to get checksum of all files

                    default: False

        Returns:
            list    -   list of output lines while executing the script.

        Raises:
            Exception:
                if any error occurred while getting the data information.

        """
        meta = kwargs.get('meta', False)
        checksum = kwargs.get('checksum', False)
        script_arguments = "-optype get -path \"{0}\"".format(data_path)

        if meta:
            script_arguments = "{0} -meta -strip".format(script_arguments)

        if checksum:
            script_arguments = "{0} -sum -strip".format(script_arguments)

        output = self._execute_jar(script_arguments)

        if output.exit_code != 0:
            raise Exception(
                "Error occurred while getting the data information {0} {1}".format(output.output, output.exception)
            )
        return output.output

    def get_items_list(
            self,
            data_path,
            sorted_output=True,
            include_parents=False):
        """Gets the list of items at the given path.

        Args:
            data_path           (str)    --  directory path
                                             to get the items list

            sorted              (bool)   --  to specify whether
                                             the list should be sorted.

                default: True

            include_parents     (bool)   --  to specify whether
                                             parent paths should be include

                default: False

        Returns:
            list    -   list of the items

        """
        output = self.get_test_data_info(data_path)
        output_list = output.split('\n')
        if include_parents:
            data_path = data_path.replace("//", "/")
            parent_list = ["/", data_path.rstrip('/')]
            for itr in range(1, data_path.count('/')):
                parent_list.append(data_path.rsplit('/', itr)[0])
            output_list.extend(parent_list)

        if sorted_output:
            output_list.sort()
        # remove empty items and return output list
        while '' in output_list:
            output_list.remove('')
        return output_list

    def remove_directory(self, directory_name):
        """Removes a directory on the hadoop client.

        Args:
            directory_name  (str)   --  name / full path of the directory to remove

        Raises:
            Exception:
                if any error occurred while removing the directory.

        """
        script_arguments = "-optype change -path \"{0}\" -delete".format(directory_name)
        output = self._execute_jar(script_arguments)

        if output.exit_code != 0:
            raise Exception(
                "Error occurred while removing the directory {0} {1}".format(output.output, output.exception)
            )

    def create_directory(self, directory_name):
        """Creates a directory on the hadoop client.

        Args:
            directory_name  (str)   --  name / full path of the directory to create

        Raises:
            Exception:
                if any error occurred while creating the directory.

        """
        script_arguments = "-optype add -path \"{0}\" -mkdir".format(directory_name)
        output = self._execute_jar(script_arguments)

        if output.exit_code != 0:
            raise Exception(
                "Error occurred while creating the directory {0} {1}".format(output.output, output.exception)
            )
